import { Random } from 'meteor/random';
import _ from 'underscore';
import s from 'underscore.string';
import _marked from 'marked';

import hljs from '../../hljs';
import { settings } from '../../../../settings';

const renderer = new _marked.Renderer();

let msg = null;

renderer.code = function(code, lang, escaped) {
	if (this.options.highlight) {
		const out = this.options.highlight(code, lang);
		if (out != null && out !== code) {
			escaped = true;
			code = out;
		}
	}

	let text = null;

	if (!lang) {
		text = `<pre><code class="code-colors hljs">${ escaped ? code : s.escapeHTML(code, true) }</code></pre>`;
	} else {
		text = `<pre><code class="code-colors hljs ${ escape(lang, true) }">${ escaped ? code : s.escapeHTML(code, true) }</code></pre>`;
	}

	if (_.isString(msg)) {
		return text;
	}

	const token = `=!=${ Random.id() }=!=`;
	msg.tokens.push({
		highlight: true,
		token,
		text,
	});

	return token;
};

renderer.codespan = function(text) {
	text = `<code class="code-colors inline">${ text }</code>`;
	if (_.isString(msg)) {
		return text;
	}

	const token = `=!=${ Random.id() }=!=`;
	msg.tokens.push({
		token,
		text,
	});

	return token;
};

renderer.blockquote = function(quote) {
	return `<blockquote class="background-transparent-darker-before">${ quote }</blockquote>`;
};

const linkRenderer = renderer.link;
renderer.link = function(href, title, text) {
	const html = linkRenderer.call(renderer, href, title, text);
	return html.replace(/^<a /, '<a target="_blank" rel="nofollow noopener noreferrer" ');
};

const highlight = function(code, lang) {
	if (!lang) {
		return code;
	}
	try {
		return hljs.highlight(lang, code).value;
	} catch (e) {
		// Unknown language
		return code;
	}
};

let gfm = null;
let tables = null;
let breaks = null;
let pedantic = null;
let smartLists = null;
let smartypants = null;

export const marked = (message) => {
	msg = message;

	if (!msg.tokens) {
		msg.tokens = [];
	}

	if (gfm == null) { gfm = settings.get('Markdown_Marked_GFM'); }
	if (tables == null) { tables = settings.get('Markdown_Marked_Tables'); }
	if (breaks == null) { breaks = settings.get('Markdown_Marked_Breaks'); }
	if (pedantic == null) { pedantic = settings.get('Markdown_Marked_Pedantic'); }
	if (smartLists == null) { smartLists = settings.get('Markdown_Marked_SmartLists'); }
	if (smartypants == null) { smartypants = settings.get('Markdown_Marked_Smartypants'); }

	msg.html = _marked(s.unescapeHTML(msg.html), {
		gfm,
		tables,
		breaks,
		pedantic,
		smartLists,
		smartypants,
		renderer,
		sanitize: true,
		highlight,
	});

	return msg;
};
