import { Meteor } from 'meteor/meteor';
import s from 'underscore.string';
import _ from 'underscore';

import { Base } from './_Base';
import Rooms from './Rooms';
import Settings from './Settings';

export class LivechatRooms extends Base {
	constructor(...args) {
		super(...args);

		this.tryEnsureIndex({ open: 1 }, { sparse: true });
		this.tryEnsureIndex({ departmentId: 1 }, { sparse: true });
		this.tryEnsureIndex({ 'metrics.chatDuration': 1 }, { sparse: true });
		this.tryEnsureIndex({ 'metrics.serviceTimeDuration': 1 }, { sparse: true });
		this.tryEnsureIndex({ 'metrics.visitorInactivity': 1 }, { sparse: true });
		this.tryEnsureIndex({ 'omnichannel.predictedVisitorAbandonmentAt': 1 }, { sparse: true });
		this.tryEnsureIndex({ closedAt: 1 }, { sparse: true });
		this.tryEnsureIndex({ servedBy: 1 }, { sparse: true });
	}

	findLivechat(filter = {}, offset = 0, limit = 20) {
		const query = Object.assign(filter, { t: 'l' });
		return this.find(query, { sort: { ts: - 1 }, offset, limit });
	}

	findOneByIdOrName(_idOrName, options) {
		const query = {
			t: 'l',
			$or: [{
				_id: _idOrName,
			}, {
				name: _idOrName,
			}],
		};

		return this.findOne(query, options);
	}

	updateSurveyFeedbackById(_id, surveyFeedback) {
		const query = {
			_id,
		};

		const update = {
			$set: {
				surveyFeedback,
			},
		};

		return this.update(query, update);
	}

	updateDataByToken(token, key, value, overwrite = true) {
		const query = {
			'v.token': token,
			open: true,
		};

		if (!overwrite) {
			const room = this.findOne(query, { fields: { livechatData: 1 } });
			if (room.livechatData && typeof room.livechatData[key] !== 'undefined') {
				return true;
			}
		}

		const update = {
			$set: {
				[`livechatData.${ key }`]: value,
			},
		};

		return this.update(query, update);
	}

	saveRoomById({ _id, topic, tags, livechatData, ...extra }) {
		const setData = { ...extra };
		const unsetData = {};

		if (topic != null) {
			if (!_.isEmpty(s.trim(topic))) {
				setData.topic = s.trim(topic);
			} else {
				unsetData.topic = 1;
			}
		}

		if (Array.isArray(tags) && tags.length > 0) {
			setData.tags = tags;
		} else {
			unsetData.tags = 1;
		}

		if (livechatData) {
			Object.keys(livechatData).forEach((key) => {
				const value = s.trim(livechatData[key]);
				if (value) {
					setData[`livechatData.${ key }`] = value;
				} else {
					unsetData[`livechatData.${ key }`] = 1;
				}
			});
		}

		const update = {};

		if (!_.isEmpty(setData)) {
			update.$set = setData;
		}

		if (!_.isEmpty(unsetData)) {
			update.$unset = unsetData;
		}

		if (_.isEmpty(update)) {
			return;
		}

		return this.update({ _id }, update);
	}

	findById(_id, fields) {
		const options = {};

		if (fields) {
			options.fields = fields;
		}

		const query = {
			t: 'l',
			_id,
		};

		return this.find(query, options);
	}

	findOneById(_id, fields) {
		const options = {};

		if (fields) {
			options.fields = fields;
		}

		const query = {
			t: 'l',
			_id,
		};

		return this.findOne(query, options);
	}

	findOneByIdAndVisitorToken(_id, visitorToken, fields) {
		const options = {};

		if (fields) {
			options.fields = fields;
		}

		const query = {
			t: 'l',
			_id,
			'v.token': visitorToken,
		};

		return this.findOne(query, options);
	}

	findOneLastServedAndClosedByVisitorToken(visitorToken, options = {}) {
		const query = {
			t: 'l',
			'v.token': visitorToken,
			closedAt: { $exists: true },
			servedBy: { $exists: true },
		};

		options.sort = { closedAt: -1 };
		return this.findOne(query, options);
	}

	findOneByVisitorToken(visitorToken, fields) {
		const options = {};

		if (fields) {
			options.fields = fields;
		}

		const query = {
			t: 'l',
			'v.token': visitorToken,
		};

		return this.findOne(query, options);
	}

	updateRoomCount = function() {
		const settingsRaw = Settings.model.rawCollection();
		const findAndModify = Meteor.wrapAsync(settingsRaw.findAndModify, settingsRaw);

		const query = {
			_id: 'Livechat_Room_Count',
		};

		const update = {
			$inc: {
				value: 1,
			},
		};

		const livechatCount = findAndModify(query, null, update);

		return livechatCount.value.value;
	}

	findOpenByVisitorToken(visitorToken, options) {
		const query = {
			t: 'l',
			open: true,
			'v.token': visitorToken,
		};

		return this.find(query, options);
	}

	findOneOpenByVisitorToken(visitorToken, options) {
		const query = {
			t: 'l',
			open: true,
			'v.token': visitorToken,
		};

		return this.findOne(query, options);
	}

	findOpenByVisitorTokenAndDepartmentId(visitorToken, departmentId, options) {
		const query = {
			t: 'l',
			open: true,
			'v.token': visitorToken,
			departmentId,
		};

		return this.find(query, options);
	}

	findByVisitorToken(visitorToken) {
		const query = {
			t: 'l',
			'v.token': visitorToken,
		};

		return this.find(query);
	}

	findByVisitorIdAndAgentId(visitorId, agentId, options) {
		const query = {
			t: 'l',
			...visitorId && { 'v._id': visitorId },
			...agentId && { 'servedBy._id': agentId },
		};

		return this.find(query, options);
	}

	findByVisitorId(visitorId) {
		const query = {
			t: 'l',
			'v._id': visitorId,
		};

		return this.find(query);
	}

	findOneOpenByRoomIdAndVisitorToken(roomId, visitorToken, options) {
		const query = {
			t: 'l',
			_id: roomId,
			open: true,
			'v.token': visitorToken,
		};

		return this.findOne(query, options);
	}

	findClosedRooms(departmentIds, options) {
		const query = {
			t: 'l',
			open: { $exists: false },
			closedAt: { $exists: true },
			...Array.isArray(departmentIds) && departmentIds.length > 0 && { departmentId: { $in: departmentIds } },
		};

		return this.find(query, options);
	}

	setResponseByRoomId(roomId, response) {
		return this.update({
			_id: roomId,
			t: 'l',
		}, {
			$set: {
				responseBy: {
					_id: response.user._id,
					username: response.user.username,
					lastMessageTs: new Date(),
				},
			},
			$unset: {
				waitingResponse: 1,
			},
		});
	}

	setNotResponseByRoomId(roomId) {
		return this.update({
			_id: roomId,
			t: 'l',
		}, {
			$set: {
				waitingResponse: true,
			},
			$unset: {
				responseBy: 1,
			},
		});
	}

	setAgentLastMessageTs(roomId) {
		return this.update({
			_id: roomId,
			t: 'l',
		}, {
			$set: {
				'responseBy.lastMessageTs': new Date(),
			},
		});
	}

	saveAnalyticsDataByRoomId(room, message, analyticsData) {
		const update = {
			$set: {},
		};

		if (analyticsData) {
			update.$set['metrics.response.avg'] = analyticsData.avgResponseTime;

			update.$inc = {};
			update.$inc['metrics.response.total'] = 1;
			update.$inc['metrics.response.tt'] = analyticsData.responseTime;
			update.$inc['metrics.reaction.tt'] = analyticsData.reactionTime;
		}

		if (analyticsData && analyticsData.firstResponseTime) {
			update.$set['metrics.response.fd'] = analyticsData.firstResponseDate;
			update.$set['metrics.response.ft'] = analyticsData.firstResponseTime;
			update.$set['metrics.reaction.fd'] = analyticsData.firstReactionDate;
			update.$set['metrics.reaction.ft'] = analyticsData.firstReactionTime;
		}

		// livechat analytics : update last message timestamps
		const visitorLastQuery = room.metrics && room.metrics.v ? room.metrics.v.lq : room.ts;
		const agentLastReply = room.metrics && room.metrics.servedBy ? room.metrics.servedBy.lr : room.ts;

		if (message.token) {	// update visitor timestamp, only if its new inquiry and not continuing message
			if (agentLastReply >= visitorLastQuery) {		// if first query, not continuing query from visitor
				update.$set['metrics.v.lq'] = message.ts;
			}
		} else if (visitorLastQuery > agentLastReply) {		// update agent timestamp, if first response, not continuing
			update.$set['metrics.servedBy.lr'] = message.ts;
		}

		return this.update({
			_id: room._id,
			t: 'l',
		}, update);
	}

	getTotalConversationsBetweenDate(t, date, { departmentId } = {}) {
		const query = {
			t,
			ts: {
				$gte: new Date(date.gte),	// ISO Date, ts >= date.gte
				$lt: new Date(date.lt),	// ISODate, ts < date.lt
			},
			...departmentId && departmentId !== 'undefined' && { departmentId },
		};

		return this.find(query).count();
	}

	getAnalyticsMetricsBetweenDate(t, date, { departmentId } = {}) {
		const query = {
			t,
			ts: {
				$gte: new Date(date.gte),	// ISO Date, ts >= date.gte
				$lt: new Date(date.lt),	// ISODate, ts < date.lt
			},
			...departmentId && departmentId !== 'undefined' && { departmentId },
		};

		return this.find(query, { fields: { ts: 1, departmentId: 1, open: 1, servedBy: 1, metrics: 1, msgs: 1 } });
	}

	getAnalyticsBetweenDate(date, { departmentId } = {}) {
		return this.model.rawCollection().aggregate([
			{
				$match: {
					t: 'l',
					ts: {
						$gte: new Date(date.gte),	// ISO Date, ts >= date.gte
						$lt: new Date(date.lt),	// ISODate, ts < date.lt
					},
					...departmentId && departmentId !== 'undefined' && { departmentId },
				},
			},
			{
				$lookup: {
					from: 'rocketchat_message',
					localField: '_id',
					foreignField: 'rid',
					as: 'messages',
				},
			},
			{
				$unwind: {
					path: '$messages',
					preserveNullAndEmptyArrays: true,
				},
			},
			{
				$group: {
					_id: {
						_id: '$_id',
						ts: '$ts',
						departmentId: '$departmentId',
						open: '$open',
						servedBy: '$servedBy',
						metrics: '$metrics',
						msgs: '$msgs',
					},
					messages: {
						$sum: {
							$cond: [{
								$and: [
									{ $ifNull: ['$messages.t', false] },
								],
							}, 1, 0],
						},
					},
				},
			},
			{
				$project: {
					_id: '$_id._id',
					ts: '$_id.ts',
					departmentId: '$_id.departmentId',
					open: '$_id.open',
					servedBy: '$_id.servedBy',
					metrics: '$_id.metrics',
					msgs: { $subtract: ['$_id.msgs', '$messages'] },
				},
			},

		]);
	}


	closeByRoomId(roomId, closeInfo) {
		const { closer, closedBy, closedAt, chatDuration, serviceTimeDuration, ...extraData } = closeInfo;

		return this.update({
			_id: roomId,
			t: 'l',
		}, {
			$set: {
				closer,
				closedBy,
				closedAt,
				'metrics.chatDuration': chatDuration,
				'metrics.serviceTimeDuration': serviceTimeDuration,
				'v.status': 'offline',
				...extraData,
			},
			$unset: {
				open: 1,
			},
		});
	}

	findOpenByAgent(userId) {
		const query = {
			t: 'l',
			open: true,
			'servedBy._id': userId,
		};

		return this.find(query);
	}

	changeAgentByRoomId(roomId, newAgent) {
		const query = {
			_id: roomId,
			t: 'l',
		};
		const update = {
			$set: {
				servedBy: {
					_id: newAgent.agentId,
					username: newAgent.username,
					ts: new Date(),
				},
			},
		};

		if (newAgent.ts) {
			update.$set.servedBy.ts = newAgent.ts;
		}

		this.update(query, update);
	}

	changeDepartmentIdByRoomId(roomId, departmentId) {
		const query = {
			_id: roomId,
			t: 'l',
		};
		const update = {
			$set: {
				departmentId,
			},
		};

		this.update(query, update);
	}

	saveCRMDataByRoomId(roomId, crmData) {
		const query = {
			_id: roomId,
			t: 'l',
		};
		const update = {
			$set: {
				crmData,
			},
		};

		return this.update(query, update);
	}

	updateVisitorStatus(token, status) {
		const query = {
			'v.token': token,
			open: true,
			t: 'l',
		};

		const update = {
			$set: {
				'v.status': status,
			},
		};

		return this.update(query, update);
	}

	removeAgentByRoomId(roomId) {
		const query = {
			_id: roomId,
			t: 'l',
		};
		const update = {
			$unset: {
				servedBy: 1,
			},
		};

		this.update(query, update);
	}

	removeByVisitorToken(token) {
		const query = {
			t: 'l',
			'v.token': token,
		};

		this.remove(query);
	}

	removeById(_id) {
		const query = {
			_id,
			t: 'l',
		};

		return this.remove(query);
	}

	setVisitorLastMessageTimestampByRoomId(roomId, lastMessageTs) {
		const query = {
			_id: roomId,
		};
		const update = {
			$set: {
				'v.lastMessageTs': lastMessageTs,
			},
		};

		return this.update(query, update);
	}

	setVisitorInactivityInSecondsById(roomId, visitorInactivity) {
		const query = {
			_id: roomId,
		};
		const update = {
			$set: {
				'metrics.visitorInactivity': visitorInactivity,
			},
		};

		return this.update(query, update);
	}
}

export default new LivechatRooms(Rooms.model, true);
