/*
 * Copyright (c) 2012-2018 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
package org.eclipse.che.ide.notification;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.ScrollPanel;
import com.google.gwt.user.client.ui.Widget;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import org.eclipse.che.ide.api.parts.base.BaseView;

/**
 * The implementation of {@link NotificationManagerView}.
 *
 * @author Andrey Plotnikov
 */
@Singleton
public class NotificationManagerViewImpl extends BaseView<NotificationManagerView.ActionDelegate>
    implements NotificationManagerView {

  interface NotificationManagerViewImplUiBinder
      extends UiBinder<Widget, NotificationManagerViewImpl> {}

  @UiField FlowPanel mainPanel;

  @UiField ScrollPanel scrollPanel;

  /** scroll events to the bottom if view is visible */
  private boolean scrollBottomRequired = false;

  /** Create view. */
  @Inject
  public NotificationManagerViewImpl(NotificationManagerViewImplUiBinder uiBinder) {
    setContentWidget(uiBinder.createAndBindUi(this));
    scrollPanel.getElement().setTabIndex(0);
  }

  @Override
  public void setContainer(NotificationContainer container) {
    mainPanel.add(container);
  }

  /** {@inheritDoc} */
  @Override
  public void scrollBottom() {
    /** scroll bottom immediately if view is visible */
    if (scrollPanel.getElement().getOffsetParent() != null) {
      scrollPanel.getElement().setScrollTop(scrollPanel.getElement().getScrollHeight());
      return;
    }

    /** otherwise, check the visibility periodically and scroll the view when it's visible */
    if (!scrollBottomRequired) {
      scrollBottomRequired = true;

      Scheduler.get()
          .scheduleFixedPeriod(
              new Scheduler.RepeatingCommand() {
                @Override
                public boolean execute() {
                  if (scrollPanel.getElement().getOffsetParent() != null) {
                    scrollPanel
                        .getElement()
                        .setScrollTop(scrollPanel.getElement().getScrollHeight());
                    scrollBottomRequired = false;
                    return false;
                  }
                  return true;
                }
              },
              1000);
    }
  }

  @Override
  protected void focusView() {
    scrollPanel.getElement().focus();
  }
}
