/*
 * Copyright (c) 2012-2018 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
package org.eclipse.che.infrastructure.docker.client.params;

import static java.util.Objects.requireNonNull;

import java.util.Objects;
import javax.validation.constraints.NotNull;
import org.eclipse.che.infrastructure.docker.client.DockerConnector;

/**
 * Arguments holder for {@link DockerConnector#killContainer(KillContainerParams)}.
 *
 * @author Mykola Morhun
 */
public class KillContainerParams {

  private String container;
  private Integer signal;

  /**
   * Creates arguments holder with required parameters.
   *
   * @param container container identifier, either id or name
   * @return arguments holder with required parameters
   */
  public static KillContainerParams create(@NotNull String container) {
    return new KillContainerParams().withContainer(container);
  }

  private KillContainerParams() {}

  /**
   * Adds container to this parameters.
   *
   * @param container container identifier, either id or name
   * @return this params instance
   * @throws NullPointerException if {@code container} is null
   */
  public KillContainerParams withContainer(@NotNull String container) {
    requireNonNull(container);
    this.container = container;
    return this;
  }

  /**
   * Adds signal code to this parameters.
   *
   * @param signal code of signal, e.g. 9 in case of SIGKILL
   * @return this params instance
   */
  public KillContainerParams withSignal(int signal) {
    this.signal = signal;
    return this;
  }

  public String getContainer() {
    return container;
  }

  public Integer getSignal() {
    return signal;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    KillContainerParams that = (KillContainerParams) o;
    return Objects.equals(container, that.container) && Objects.equals(signal, that.signal);
  }

  @Override
  public int hashCode() {
    return Objects.hash(container, signal);
  }
}
