/*
 * Copyright (c) 2012-2018 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
package org.eclipse.che.selenium.stack;

import static org.eclipse.che.commons.lang.NameGenerator.generate;
import static org.eclipse.che.selenium.core.constant.TestBuildConstants.BUILD_SUCCESS;
import static org.eclipse.che.selenium.core.constant.TestCommandsConstants.BUILD_COMMAND;
import static org.eclipse.che.selenium.core.constant.TestIntelligentCommandsConstants.CommandItem.BUILD_COMMAND_ITEM;
import static org.eclipse.che.selenium.core.constant.TestIntelligentCommandsConstants.CommandItem.RUN_COMMAND_ITEM;
import static org.eclipse.che.selenium.core.constant.TestProjectExplorerContextMenuConstants.ContextMenuCommandGoals.BUILD_GOAL;
import static org.eclipse.che.selenium.core.constant.TestProjectExplorerContextMenuConstants.ContextMenuCommandGoals.RUN_GOAL;
import static org.eclipse.che.selenium.pageobject.dashboard.NewWorkspace.Stack.ANDROID;

import com.google.common.collect.ImmutableList;
import com.google.inject.Inject;
import java.util.List;
import org.eclipse.che.selenium.core.client.TestWorkspaceServiceClient;
import org.eclipse.che.selenium.core.user.DefaultTestUser;
import org.eclipse.che.selenium.core.workspace.TestWorkspace;
import org.eclipse.che.selenium.pageobject.Consoles;
import org.eclipse.che.selenium.pageobject.Ide;
import org.eclipse.che.selenium.pageobject.ProjectExplorer;
import org.eclipse.che.selenium.pageobject.dashboard.CreateWorkspaceHelper;
import org.eclipse.che.selenium.pageobject.dashboard.Dashboard;
import org.testng.annotations.AfterClass;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;

/** @author Skoryk Serhii */
public class CreateWorkspaceFromAndroidStackTest {

  private static final String WORKSPACE_NAME = generate("workspace", 4);
  private static final String MOBILE_ANDROID_HELLO_WORLD = "mobile-android-hello-world";
  private static final String MOBILE_ANDROID_SIMPLE = "mobile-android-simple";

  private List<String> projects =
      ImmutableList.of(MOBILE_ANDROID_HELLO_WORLD, MOBILE_ANDROID_SIMPLE);

  @Inject private Ide ide;
  @Inject private Consoles consoles;
  @Inject private Dashboard dashboard;
  @Inject private DefaultTestUser defaultTestUser;
  @Inject private ProjectExplorer projectExplorer;
  @Inject private CreateWorkspaceHelper createWorkspaceHelper;
  @Inject private TestWorkspaceServiceClient workspaceServiceClient;

  // it is used to read workspace logs on test failure
  private TestWorkspace testWorkspace;

  @BeforeClass
  public void setUp() {
    dashboard.open();
  }

  @AfterClass
  public void tearDown() throws Exception {
    workspaceServiceClient.delete(WORKSPACE_NAME, defaultTestUser.getName());
  }

  @Test
  public void checkWorkspaceCreationFromAndroidStack() {
    // store info about created workspace to make SeleniumTestHandler.captureTestWorkspaceLogs()
    // possible to read logs in case of test failure
    testWorkspace =
        createWorkspaceHelper.createWorkspaceFromStackWithProjects(
            ANDROID, WORKSPACE_NAME, projects);

    ide.switchToIdeAndWaitWorkspaceIsReadyToUse();

    projectExplorer.waitProjectInitialization(MOBILE_ANDROID_HELLO_WORLD);
    projectExplorer.waitProjectInitialization(MOBILE_ANDROID_SIMPLE);
    consoles.waitJDTLSProjectResolveFinishedMessage(
        MOBILE_ANDROID_HELLO_WORLD, MOBILE_ANDROID_SIMPLE);
  }

  @Test(priority = 1)
  public void checkMobileAndroidHelloWorldProjectCommands() {
    consoles.executeCommandFromProjectExplorer(
        MOBILE_ANDROID_HELLO_WORLD, BUILD_GOAL, BUILD_COMMAND, BUILD_SUCCESS);

    consoles.executeCommandFromProjectExplorer(
        MOBILE_ANDROID_HELLO_WORLD,
        RUN_GOAL,
        RUN_COMMAND_ITEM.getItem(MOBILE_ANDROID_HELLO_WORLD),
        BUILD_SUCCESS);
  }

  @Test(priority = 1)
  public void checkMobileAndroidSimpleProjectCommands() {
    consoles.executeCommandFromProjectExplorer(
        MOBILE_ANDROID_SIMPLE,
        RUN_GOAL,
        BUILD_COMMAND_ITEM.getItem(MOBILE_ANDROID_SIMPLE),
        BUILD_SUCCESS);
  }
}
